use warnings;
use strict;
use Getopt::Long;

Getopt::Long::Configure ("bundling");

my $debug = 0;
my $quiet = 0;
my $usage = 0;
my $help = 0;

GetOptions ('debug|d' => \$debug,
	'help|h' => \$help,
	'usage|u' => \$usage,
	'quiet|q' => \$quiet );

if ($debug && $quiet) {
	$debug = 1;
	$quiet = 0;
	print "Both debug and quiet options enabled.  Debug will take precedence.\n";
}

sub print_usage () {
	print "\n";
	print "Usage: $0 [-dhquv] FileName SummaryFile OnsetFile MaxSpsFile MinSpsFile\n";
	print "   -d = debug mode (verbose)\n";
	print "   -h = print help and exit\n";
	print "   -q = quiet mode (no output of any kind).\n";
	print "   -u = print this usage guide\n";
	print "   -v = debug mode (verbose)\n";
	print "Note that (-d / -v) and (-q) are mutually exclusive\n";
	print "FileName is the name of the file to be parsed\n";
	print "\n";
	exit 0;
}

sub print_help () {
	print "\n";
	print "*******************************************************************************\n";
	print "$0 is a script to parse a print file created by one\n";
	print "    of the 'make_____Runs.sh' scripts, and summarize the properties of those  \n";
	print "   runs that contain garnet to a set of CSV (comma-separated values) files that \n";
	print "   can be directly input into Excel.\n";
	print " \n";
	print "Note that $0 makes thinRuns.sh and thinRuns.pl obsolete, \n";
	print "   by incorporating their functionality.\n";
	print "*******************************************************************************\n";
	print "\n";
	exit 0;
}

sub inarray ($$) {
	my $arrayRef = $_[0];
	my $item = $_[1];
	foreach my $element (@$arrayRef) {
		if ($item == $element) {
			return 1;
		}
	}
	return 0;
}

sub indexOf ($$) {
	my $arrayRef = $_[0];
	my $item = $_[1];
	my $index = 0;
	foreach my $element (@$arrayRef) {
		if ($item == $element) {
			return $index;
		}
		$index++;
	}
	return -1;
}

sub thinFile ($) {
	my $filename = shift(@_);
	my $source = "$filename.bak";
	rename ($filename, $source) or die "Can't rename the file $filename.  Check directory permissions.\n";
	my $dest = "$filename";

	if ($debug) { print "In thinFile.  Source:|$source|  Dest:|$dest|.\n"; }
	
	open IN, $source or die "Can't read source file $source: $!\n";
	open OUT, ">$dest" or die "Can't write to destination file $dest: $!\n";
	
	my $line = "";
	
	do {
		$line = <IN>;
		print OUT $line;
	} while ($line !~ /Phases and \(projected\) compositions/);
	
	print OUT "\n[composition list thinned out here]\n";
	
	do {
		$line = <IN>;
	} while ($line !~ /Excluded phases/);
	
	print OUT $line;
	
	while ($line = <IN>) {
		print OUT $line;
	};
	
	close OUT;
	close IN;

	unlink $source;
}

if ($help) { print_help(); }
if ($usage) { print_usage(); }

if (@ARGV == 0) { print_usage(); }

my $source = shift @ARGV;
my $summary = shift @ARGV;
my $onset = shift @ARGV;
my $maxSpsFile = shift @ARGV;
my $minSpsFile = shift @ARGV;

thinFile($source) or die "Can't thin source file $source: $!\n";

open IN, $source or die "Can't read source file $source: $!\n";

open OUTSUMMARY, ">>$summary" or die "Can't write to destination file $summary: $!\n";	
open OUTONSET, ">>$onset" or die "Can't write to destination file $onset: $!\n";	
open OUTMAXSPS, ">>$maxSpsFile" or die "Can't write to destination file $maxSpsFile: $!\n";
open OUTMINSPS, ">>$minSpsFile" or die "Can't write to destination file $minSpsFile: $!\n";

$source =~ /(.*)_(\d*)print\.txt/;
my $runNum = $2;

my @lines = <IN>;
my $numlines = @lines;

my $curline;

my @maxes = (0,0,0,0,0,0,0,0);	#sps, alm, pyp, grs, conditionnum, P, T, X
my @mins = (99,0,0,0,0,0,0,0);	#sps, alm, pyp, grs, conditionnum, P, T, X
my @temps = ();
my @co2s = ();
my @grts = ();

for (my $linenum = 0; $linenum < $numlines; $linenum++) {
	if ($lines[$linenum] =~ /\(( *)(\d*)\) Stable phases at/ ) {
		my $condNum = $2;
		
		$linenum++;
		$lines[$linenum] =~ /=(\s*)(\S*)/;
		my $pressure = $2;

		$linenum++;
		$lines[$linenum] =~ /=(\s*)(\S*)/;
		my $temp = $2;
		if (!(inarray (\@temps, $temp))) {
			push(@temps, $temp);
		}

		$linenum++;
		$lines[$linenum] =~ /=(\s*)(\S*)/;
		my $co2 = $2;
		if (!(inarray (\@co2s, $co2))) {
			push(@co2s, $co2);
		}

		$linenum++;	#skip title line
		$linenum++;	#skip column header line

		# Examine all molar phase composition lines
		my @grtlines;
		do {
			if ($lines[$linenum] =~ /GrPyAlSp\(G|GrPyAlSp\(B|Gt\(HP\)|Gt\(stx\)|Gt\(EWHP\)/) {
				push(@grtlines, $lines[$linenum]);
			}
			$linenum++;
		} while ($lines[$linenum] !~ /Molar Properties/);

		my @comps = (0,0,0,0); 	#sps, alm, pyp, grs
		my $moles = 0;
		
		# If we've got any garnet lines
		if (@grtlines > 0) {
			# do something to calculate molar average proerties here
			my $grtlines = @grtlines;
			# extract and average multiple values (will also work with a single garnet value)
			$moles = 0;
			my $newmoles = 0;
			foreach my $grt (@grtlines) {
				my @values = split(/ +/, $grtlines[0]);
				$newmoles = $values[1];
				$comps[0] = (($comps[0] * $moles) + ($values[11] * $newmoles)) / ($moles + $newmoles);
				$comps[1] = (($comps[1] * $moles) + ($values[12] * $newmoles)) / ($moles + $newmoles);
				$comps[2] = (($comps[2] * $moles) + ($values[5] * $newmoles)) / ($moles + $newmoles);
				$comps[3] = (($comps[3] * $moles) + ($values[9] * $newmoles)) / ($moles + $newmoles);
				$moles += $newmoles;
			}
			for (@comps) {$_ /= 3.0}
			if ($comps[0] > $maxes[0]) {
				@maxes[0..3] = @comps;
				$maxes[4] = $condNum;
				$maxes[5] = $pressure;
				$maxes[6] = $temp;
				$maxes[7] = $co2;
			}
			if ($comps[0] < $mins[0]) {
				@mins[0..3] = @comps;
				$mins[4] = $condNum;
				$mins[5] = $pressure;
				$mins[6] = $temp;
				$mins[7] = $co2;
			}
		} # if we've got garnet lines

		# skip Molar Properties lines
		do {
			$linenum++;
		} while ($lines[$linenum] !~ /Bulk Composition/);

		# skip BC line, 2 blank lines, and heading line
		$linenum++;
		$linenum++;
		$linenum++;
		$linenum++;
		# collect bulk composition data; the algorithm could be simplified if I were sure that
		# the oxides were always present and in the same order.
		my @wtpct = (0,0,0,0,0,0,0,0,0,0,0); # Na2O, MgO, Al2O3, SiO2, K2O, CaO, TiO2, MnO, FeO, H2O, CO2
		do {
			if ($debug) { print "The line:$lines[$linenum]\n"; }
			$lines[$linenum] =~ s/([A-Z,1-3]*)( +)([0-9,\.]+)( +)([0-9,\.]+)( +)([0-9,\.]+)( *)\n/$1$6$7/g;
			if ($debug) { print "The new line:$lines[$linenum]\n"; }
			if ($debug) { print "Item 1 and 7: $1|$7\n"; }
			my $thiswtpct = $7;
			
			if ($1 =~ /NA2O/) {
				$wtpct[0] = $thiswtpct;
			} elsif ($1 =~ /MGO/) {
				$wtpct[1] = $thiswtpct;
			} elsif ($1 =~ /AL2O3/) {
				$wtpct[2] = $thiswtpct;
			} elsif ($1 =~ /SIO2/) {
				$wtpct[3] = $thiswtpct;
			} elsif ($1 =~ /K2O/) {
				$wtpct[4] = $thiswtpct;
			} elsif ($1 =~ /CAO/) {
				$wtpct[5] = $thiswtpct;
			} elsif ($1 =~ /TIO2/) {
				$wtpct[6] = $thiswtpct;
			} elsif ($1 =~ /MNO/) {
				$wtpct[7] = $thiswtpct;
			} elsif ($1 =~ /FEO/) {
				$wtpct[8] = $thiswtpct;
			} elsif ($1 =~ /H2O/) {
				$wtpct[9] = $thiswtpct;
			} elsif ($1 =~ /CO2/) {
				$wtpct[10] = $thiswtpct;
			}
			$linenum++;
		} while ($lines[$linenum] !~ /^$/);
		
		if ($debug) { print "Wt pcts:\n"; }
		if ($debug) { print "$wtpct[0],$wtpct[1],$wtpct[2],$wtpct[3],$wtpct[4],\n"; }

		#store separator and set to comma
		my $oldSep = $";
		$" = ",";
		
		#print out data here to summary file
		my $compStr = "@comps";
		my $wtpctStr = "@wtpct";
		
		if (@grtlines > 0) {
			print OUTSUMMARY "$runNum,$condNum,$pressure,$temp,$co2,$compStr,$wtpctStr\n";
			
			#copy this garnet to the array for winnowing below
			my @oneGrt = ($runNum, $condNum, $pressure, $temp, $co2, $compStr, $wtpctStr);
			push (@grts, [ @oneGrt ]);
		}
		
		$" = $oldSep;
		
	}
}

print OUTMAXSPS "$runNum,$maxes[4],$maxes[5],$maxes[6],$maxes[7],$maxes[0],$maxes[1],$maxes[2],$maxes[3]\n";
print OUTMINSPS "$runNum,$mins[4],$mins[5],$mins[6],$mins[7],$mins[0],$mins[1],$mins[2],$mins[3]\n";


close OUTMAXSPS;
close OUTMINSPS;
close OUTSUMMARY;

@temps = sort(@temps);
@co2s = sort(@co2s);

if ($debug) { print OUTONSET "Temps: @temps\n CO2s: @co2s\n"; }

# make a 2-D array of -1's, with the first dimension being XCo2s and the second dimension being Ts
my @XTarray;
my $numXs = @co2s;
my $numTs = @temps;
for my $thisX (0 .. $numXs-1 ) {
	my @Tarray;
	for my $thisT (0 .. $numTs-1 ) {
		push (@Tarray, -1);
	}
	push (@XTarray, [ @Tarray ] );
}

# go through each grt and change the T-X spot to the index of that garnet in the array (garnet is stable there)
my $index = 0;
foreach my $thisGrt (@grts) {
	my $thisCO2index = indexOf(\@co2s, $$thisGrt[4]);
	my $thisTindex = indexOf(\@temps, $$thisGrt[3]);
	$XTarray[ $thisCO2index ][ $thisTindex ] = $index;
	$index++;
}


for my $thisCO2 ( 0 .. $#co2s) {
	#in each CO2 value, look for the first Temp that has garnet
	my $grtFound = 0;	# we haven't yet found a garnet in this CO2 value
	for my $thisTemp ( 0 .. $#temps) {
		my $grtNum = $XTarray[ $thisCO2 ][ $thisTemp ];
		if ($debug) { print "The garnet number is: $grtNum\n"; }
		if ( ( $grtNum > -1 ) && (! $grtFound) ) {
			my $thisGrt = $grts[ $grtNum ];
			print OUTONSET "$$thisGrt[0],$$thisGrt[1],$$thisGrt[2],$$thisGrt[3],$$thisGrt[4],$$thisGrt[5],$$thisGrt[6],\n";
			$grtFound = 1;
		}
	}
}


close OUTONSET;

exit 0;