#!/bin/bash

NO_ARGS=0 
E_OPTERROR=65

print_usage () {
	echo
	echo "Usage: `basename $0` [-dhquv] [-sStartNum] [-eEndNum] RunDirectory RunName"
	echo '   -d = debug mode (verbose)'
	echo '   -h = print help and exit'
	echo '   -q = quiet mode (no output of any kind).'
	echo '   -u = print this usage guide'
	echo '   -v = debug mode (verbose)'
	echo 'Note that (-d / -v) and (-q) are mutually exclusive'
	echo 'RunName is the name of the template file that precedes "template.txt.'
	echo 'RunDirectory is the relative path from the current directory to the one '
	echo '   where the runs are located.'
}


if [ $# -eq "$NO_ARGS" ]; then  # Script invoked with no command-line args?
	print_usage
	exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
fi  

while getopts ":dehn:qsuv" Option
do
  case $Option in
	d | v ) 
		DEBUG=1
		QUIET=
		;;
	e )
		ENDNUM=$OPTARG
		;;
	h )
		clear
		echo "*******************************************************************************"
		echo '`basename $0` is a script to parse the resulting set of print files created by '
		echo '   one of the "make_____Runs.sh" scripts, and summarize the properties of'
		echo '   those runs that contain garnet.  These summaries are saved as a set of'
		echo '   CSV (comma-separated values) files that can be directly input into Excel.'
		echo
		echo 'If you omit the start and end parameters, then all suitable files will be '
		echo '   parsed.'
		echo
		echo 'This script handles the files, but the identically-names perl script does '
		echo '   the actual parsing of each individual file.'
		echo 
		echo "*******************************************************************************"
		echo
		exit
		;;
	q )
		DEBUG=
		QUIET=1
		;;
	s )
		STARTNUM=$OPTARG
		;;
	u )
		print_usage
		exit
		;;
	* )
		echo
		echo "One or more options were not recognized or arguments were not supplied to "
		echo "    options that require them.  Run the script with no arguments for a usage"
		echo "    guide."
		echo
		echo
		exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
		;;
  esac
done


if [[ ! $QUIET ]]; then
	echo "********************************************************************"
	echo -e "Script: \033[1m`basename $0`\033[0m"
	echo "Version 1.3"
	echo "Date: September 5, 2006"
	echo "This script was written by David Hirsch (http://www.davehirsch.com)."
	echo "Post, redistribute, and alter freely so long as this credit is "
	echo "retained and no money is charged for access to the script."
	echo "********************************************************************"
	echo "Use `basename $0` -h for help"
fi

DEFSTARTNUM=1
# Was the number of runs set on the command line?
if [[ ! -n "$STARTNUM" ]]; then
	STARTNUM=$DEFSTARTNUM
	if [[ ! $QUIET ]]; then
		echo "Using default start number: $STARTNUM"
	fi
else
	if [[ $DEBUG ]]; then
		echo "Using start number: $STARTNUM"
	fi
fi

if [[ -n "$ENDNUM" ]]; then
	if [[ ! $QUIET ]]; then
		echo "No end number specified; will go until files run out."
	fi
else
	if [[ $DEBUG ]]; then
		echo "Using no preset end number."
	fi
fi

shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.


if [ -d "$1" ]
# Test if command line argument present (non-empty).
then
	FULLDIRNAME="$1"
else
	echo "Directory not specified"
	exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
fi

if [ -n "$2" ]
# Test for short run name, prefix to each file.
then
	RUNNAME="$2"
else
	echo "The second argument must be the short run name, prefix to the 'template.txt'."
	exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
fi

cd "$FULLDIRNAME"

SUMMARYFILE=$RUNNAME"_GarnetSummary.csv"
ONSETFILE=$RUNNAME"_GarnetOnsets.csv"
MAXSPSFILE=$RUNNAME"_MaxSps.csv"
MINSPSFILE=$RUNNAME"_MinSps.csv"

if [[ $DEBUG ]]; then echo "We are in |$PWD|"; fi

RUNNUM=$STARTNUM

if [[ $DEBUG ]]; then echo "Params: |$STARTNUM|$RUNNUM|$ENDNUM|$FULLDIRNAME|"; fi

FILENAME="${RUNNAME}_${RUNNUM}print.txt"

if [[ $DEBUG ]]; then echo "Filename: |$FILENAME|"; fi
#

if [[ -e "$SUMMARYFILE" ]]; then rm "$SUMMARYFILE"; fi
if [[ -e "$ONSETFILE" ]]; then rm "$ONSETFILE"; fi
if [[ -e "$MAXSPSFILE" ]]; then rm "$MAXSPSFILE"; fi
if [[ -e "$MINSPSFILE" ]]; then rm "$MINSPSFILE"; fi

printf "Detailed Garnet composition output from $FILENAME and subsequent files\nRun Number,Condition Number,P(bar),T(K),X(CO2),X(sps),X(alm),X(pyp),X(grs),Wt%% Na2O,Wt%% Mg2O,Wt%% Al2O3,Wt%% SiO2,Wt%% K2O,Wt%% CaO,Wt%% TiO2,Wt%% MnO,Wt%% FeO,Wt%% H2O,Wt%% CO2\n" > "$SUMMARYFILE"
printf "Detailed Garnet onset composition output from $FILENAME and subsequent files\nRun Number,Condition Number,P(bar),T(K),X(CO2),X(sps),X(alm),X(pyp),X(grs),Wt%% Na2O,Wt%% Mg2O,Wt%% Al2O3,Wt%% SiO2,Wt%% K2O,Wt%% CaO,Wt%% TiO2,Wt%% MnO,Wt%% FeO,Wt%% H2O,Wt%% CO2\n" > "$ONSETFILE"
printf "Maximum Sps from $FILENAME and subsequent files\nRun Number,Condition Number,P(bar),T(K),X(CO2),X(sps),X(alm),X(pyp),X(grs)\n" > "$MAXSPSFILE"
printf "Minimum Sps from $FILENAME and subsequent files\nRun Number,Condition Number,P(bar),T(K),X(CO2),X(sps),X(alm),X(pyp),X(grs)\n" > "$MINSPSFILE"

if [[ -n "$ENDNUM" ]]; then
	for ((RUNNUM=STARTNUM; RUNNUM <= ENDNUM ; RUNNUM++))
	do
		if [ -e "$FILENAME" ]; then
			if [[ ! $QUIET ]]; then echo "Getting garnet composition from: $FILENAME"; fi
			perl -w "$PWD/../getGrtComps.pl" "$PWD/$FILENAME" "$PWD/$SUMMARYFILE" "$PWD/$ONSETFILE" "$PWD/$MAXSPSFILE" "$PWD/$MINSPSFILE"
			(( RUNNUM += 1 ))
			FILENAME="${RUNNAME}_${RUNNUM}print.txt"
		fi
	done
else
	while [[ -e $FILENAME ]]; do
		if [[ ! $QUIET ]]; then echo "Getting garnet composition from: $FILENAME"; fi
		perl -w "$PWD/../getGrtComps.pl" "$PWD/$FILENAME" "$PWD/$SUMMARYFILE" "$PWD/$ONSETFILE" "$PWD/$MAXSPSFILE" "$PWD/$MINSPSFILE"
		(( RUNNUM += 1 ))
		FILENAME="${RUNNAME}_${RUNNUM}print.txt"
	done
fi


exit 0