#!/bin/bash

#############################################################
#
#     IMPORTANT!!
#
# IF YOU ARE RUNNING THIS ON A NON-MACOSX 10.4 SYSTEM, YOU SHOULD CHECK TO SEE IF THE GET_RANDOM 
#	ALGORITHM WORKS AS EXPECTED.  THE CURRENT FORMULATION RELIES ON THE MAXIMUM "SHORT" (CREATED BY
#	/dev/urandom) BEING 32768.
# IF YOU ARE ON A DIFFERENT SYSTEM, YOU SHOULD CHANGE THIS VALUE AS APPROPRIATE.

MAXSHORT=32768

#############################################################

NO_ARGS=0 
E_OPTERROR=65

print_usage () {
	echo
	echo "Usage: `basename $0` [-dhquv] [-nNumberOfRuns] [-sStartNum] RunName"
	echo '   -d = debug mode (verbose)'
	echo '   -h = print help and exit'
	echo '   -q = quiet mode (no output of any kind, although vertex will have output).'
	echo '   -u = print this usage guide'
	echo '   -v = debug mode (verbose)'
	echo 'Note that (-d / -v) and (-q) are mutually exclusive'
	echo 'RunName is the name of the template file that precedes "template.txt.'
	echo
}


if [ $# -eq "$NO_ARGS" ]; then  # Script invoked with no command-line args?
	print_usage
	exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
fi  

while getopts ":dhn:quv" Option
do
  case $Option in
	d | v ) 
		DEBUG=1
		QUIET=
		;;
	h )
		clear
		echo "*******************************************************************************"
		echo '`basename $0` is a script to take a special template file  '
		echo '   that includes a setof stability calculations at discrete P-T-X points, '
		echo '   create a set of build files based on that template that vary randomly in   '
		echo '   their bulk compopsition (within specified limits), and then parse the   '
		echo '   resulting set of print filesfor the garnet compositions, which are summarized'
		echo '   in a set of CSV files.'
		echo
		echo 'You need to edit the script in advance to specify the range of randomness for  '
		echo '   each element.' 
		echo
		echo 'The perplex program vertex, as well as its ancillary data files,'
		echo '   must be in the same directory as this script and the template file.'
		echo
		echo 'If you omit the number of runs parameter, then the default of 100 will'
		echo "  be used."
		echo 
		echo "*******************************************************************************"
		echo
		exit
		;;
	n )
		NUMRUNS=$OPTARG
		;;
	q )
		DEBUG=
		QUIET=1
		;;
	s )
		STARTNUM=$OPTARG
		;;
	u )
		print_usage
		exit
		;;
	* )
		echo
		echo "One or more options were not recognized or arguments were not supplied to "
		echo "    options that require them.  Run the script with no arguments for a usage"
		echo "    guide."
		echo
		echo
		exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
		;;
  esac
done


if [[ ! $QUIET ]]; then
	echo "********************************************************************"
	echo -e "Script: \033[1m`basename $0`\033[0m"
	echo "Version 1.4"
	echo "Date: September 7, 2006"
	echo "This script was written by David Hirsch (http://www.davehirsch.com)."
	echo "Post, redistribute, and alter freely so long as this credit is "
	echo "retained and no money is charged for access to the script."
	echo "********************************************************************"
	echo "Use `basename $0` -h for help"
fi

DEFNUMRUNS=100

# Was the number of runs set on the command line?
if [[ ! -n "$NUMRUNS" ]]; then
	NUMRUNS=$DEFNUMRUNS	#default grt solution model
	if [[ ! $QUIET ]]; then
		echo "Using default number of runs: $NUMRUNS"
	fi
else
	if [[ $DEBUG ]]; then
		echo "Using number of runs: $NUMRUNS"
	fi
fi

DEFSTARTNUM=1
# Was the number of runs set on the command line?
if [[ ! -n "$STARTNUM" ]]; then
	STARTNUM=$DEFSTARTNUM
	if [[ ! $QUIET ]]; then
		echo "Using default start number: $STARTNUM"
	fi
else
	if [[ $DEBUG ]]; then
		echo "Using start number: $STARTNUM"
	fi
fi


shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

if [[ -n "$1" ]]; then
	RUNNAME=$1
else  
	echo "Need to specify the short run name, prefix to 'template.txt'."
	exit $E_OPTERROR
fi

ROOT=`pwd`

if [ ! -e "$ROOT/vertex" ]		# test whether vertex exists here
then
	echo "vertex not found in directory $ROOT.  Quitting `basename $0`."
	exit
fi

if [ ! -e "$ROOT/hp02ver.dat" ]		# test whether vertex exists here
then
	echo "hp02ver.dat not found in directory $ROOT.  Quitting `basename $0`."
	exit
fi

if [ ! -e "$ROOT/newest_format_solut.dat" ]		# test whether vertex exists here
then
	echo "newest_format_solut.dat not found in directory $ROOT.  Quitting `basename $0`."
	exit
fi

TEMPLATENAME=$RUNNAME"template.txt"
if [ ! -e "$ROOT/$TEMPLATENAME" ]		# test whether template exists here
then
	echo "Template file: $TEMPLATENAME was not found in directory $ROOT.  Quitting `basename $0`."
	exit
fi

DIRNAME=$RUNNAME"runs"
DATESTR=`date +%y%m%d_%H%M`
FULLDIRNAME="$DIRNAME$DATESTR"

mkdir "$ROOT/$FULLDIRNAME"

cp vertex $FULLDIRNAME
cp hp02ver.dat $FULLDIRNAME
cp newest_format_solut.dat $FULLDIRNAME
cp $TEMPLATENAME $FULLDIRNAME

cd "$ROOT/$FULLDIRNAME"

# Measured bulk composition for sample # 711:
# NA2O=0.00000
# MGO=4.65500
# AL2O3=12.7030
# SIO2=55.4845
# K2O=3.21475
# CAO=4.15235
# TIO2=0.620750
# MNO=1.97280
# FEO=9.99485

# Upper and lower bounds of random values for bulk composition:
#ELEMENT:  Na    Mg    Al    Si    K     Ca    Ti    Mn    Fe
TOPS=(     7     9     15    60    9     9     2     8     14  )
BOTTOMS=(  0.01  0.01   8    50    1     1     0.01  0.2    7  )


get_random ()	# takes two arguments, lower ($1) and upper ($2)
{
	SEED=$(head -1 /dev/urandom | od -tuS -N 1 | awk '{ print $2 }')
	RANDOM=$SEED
	rnum=$(echo "scale=9; ($RANDOM * ($2 - $1) / $MAXSHORT ) + $1" | bc)
	echo $rnum
}

ordinal () #takes integer argument (e.g., 34), returns ordinal string (e.g., "34th")
{
	LASTDIGIT=$(echo "$1 % 10" | bc)
	case $LASTDIGIT in
		0 | 4 | 5 | 6 | 7 | 8 | 9 )
			SUFFIX="th"
			;;
		1 )
			SUFFIX="st"
			;;
		2 )
			SUFFIX="nd"
			;;
		3 )
			SUFFIX="rd"
			;;
	esac
	echo "${1}${SUFFIX}"
}

for ((RUNNUM=STARTNUM; RUNNUM <= NUMRUNS ; RUNNUM++))
do
	if [[ ! $QUIET ]]; then
		RUNORD=$(ordinal $RUNNUM)
		let "COUNT = $RUNNUM - $STARTNUM + 1"
		SEQORD=$(ordinal $COUNT)
		echo "Starting $RUNORD run, numbered $SEQORD out of $NUMRUNS."
	fi
	NAVAL=$(get_random ${BOTTOMS[0]} ${TOPS[0]})
	MGVAL=$(get_random ${BOTTOMS[1]} ${TOPS[1]})
	ALVAL=$(get_random ${BOTTOMS[2]} ${TOPS[2]})
	SIVAL=$(get_random ${BOTTOMS[3]} ${TOPS[3]})
	KVAL=$(get_random ${BOTTOMS[4]} ${TOPS[4]})
	CAVAL=$(get_random ${BOTTOMS[5]} ${TOPS[5]})
	TIVAL=$(get_random ${BOTTOMS[6]} ${TOPS[6]})
	MNVAL=$(get_random ${BOTTOMS[7]} ${TOPS[7]})
	FEVAL=$(get_random ${BOTTOMS[8]} ${TOPS[8]})
	sed -e 's/RUNNUM/'"$RUNNUM"'/g' \
		-e 's/RUNNAME/'"$RUNNAME"'/g' \
		-e 's/NA/'"$NAVAL"'/g'\
		-e 's/MG/'"$MGVAL"'/g'\
		-e 's/AL/'"$ALVAL"'/g'\
		-e 's/SI/'"$SIVAL"'/g'\
		-e 's/K/'"$KVAL"'/g'\
		-e 's/CA/'"$CAVAL"'/g'\
		-e 's/TI/'"$TIVAL"'/g'\
		-e 's/MN/'"$MNVAL"'/g'\
		-e 's/FE/'"$FEVAL"'/g'\
		"$TEMPLATENAME" > "${RUNNAME}_Run_$RUNNUM.txt"
	echo "${RUNNAME}_Run_$RUNNUM.txt" > "input.txt"
	./vertex < "input.txt"
	rm input.txt
done

cd ".."

if [[ $DEBUG ]]; then pwd; fi

if [[ $DEBUG ]]; then OPTIONSTR="${OPTIONSTR}-d "; fi
if [[ $QUIET ]]; then OPTIONSTR="${OPTIONSTR}-q "; fi

./getGrtComps.sh "${OPTIONSTR} -s${STARTNUM} $ROOT/$FULLDIRNAME $RUNNAME"


# CLEAN UP
rm "${ROOT}/${FULLDIRNAME}/vertex"
rm "${ROOT}/${FULLDIRNAME}/hp02ver.dat"
rm "${ROOT}/${FULLDIRNAME}/newest_format_solut.dat"
rm "${ROOT}/${FULLDIRNAME}/pseudocompound_glossary.dat"
rm "${ROOT}/${FULLDIRNAME}/${TEMPLATENAME}"

exit 0