#!/bin/bash

NO_ARGS=0 
E_OPTERROR=65



print_usage () {
	echo
	echo -e "Usage:\033[1m `basename $0` [-dhpquv] [-g'GrtSolutionModel'] BuildFile \033[0m"
	echo '   -d = debug mode (verbose)'
	echo '   -h = print help and exit'
	echo '   -p = do plots only without running vertex'
	echo '   -q = quiet mode (no output of any kind, although vertex and '
	echo '        psvdraw will have output).'
	echo '   -u = print this usage guide'
	echo '   -v = debug mode (verbose)'
	echo 'Note that (-d / -v) and (-q) are mutually exclusive'
	echo
}


if [ $# -eq "$NO_ARGS" ]; then  # Script invoked with no command-line args?
	print_usage
	exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
fi  

while getopts ":dg:hpquv" Option
do
  case $Option in
    d | v ) 
    	DEBUG=1
    	QUIET=
    	;;
    g )
    	GRTNAME=$OPTARG
    	;;
    h )
		clear
		echo "*******************************************************************************"
		echo "`basename $0` is a script to start with a bulid file and automatically run "
		echo "  vertex and psvdraw to make a set of desired postscript diagrams for plotting "
		echo "  various pseudosection properties."
		echo
		echo 'You need to edit the script in advance to specify the type of plot(s) you wish'
		echo '   to make (these are near the bottom of the script).'
		echo
		echo 'The perplex programs vertex and psvdraw, as well as their ancillary data files,'
		echo '   must be in the same directory as this script and the build file.'
		echo
		echo "Any plot file name specified in the build file will be replaced by a name"
		echo "   derived from the build file name."
		echo
		echo 'If you omit the GrtSolutionModel parameter, then the default of "Gt(HP)" will'
		echo "  be used."
		echo 
		echo "Note that a common error occurs when specifying solution models, because the"
		echo "  parentheses must be preceded by a backslash '\', or the name enclosed in "
		echo "  single quotes."
		echo "*******************************************************************************"
		echo
		exit
		;;
	p )
		NOVERTEX=1
		;;
	q )
		DEBUG=
		QUIET=1
		;;
	u )
		print_usage
		exit
		;;
	* )
		echo
		echo "One or more options were not recognized or arguments were not supplied to "
		echo "    options that require them.  Run the script with no arguments for a usage"
		echo "    guide."
		echo
		echo
		exit $E_OPTERROR        # Exit and explain usage, if no argument(s) given.
		;;
  esac
done


if [[ ! $QUIET ]]; then
	echo "********************************************************************"
	echo -e "Script: \033[1m`basename $0`\033[0m"
	echo "Version 1.7"
	echo "Date: September 6, 2006"
	echo "This script was written by David Hirsch (http://www.davehirsch.com)."
	echo "Post, redistribute, and alter freely so long as this credit is "
	echo "retained and no money is charged for access to the script."
	echo "********************************************************************"
	echo "Use `basename $0` -h for help"
fi

# Was the garnet solution model set on the command line?
if [[ ! -n "$GRTNAME" ]]; then
	GRTNAME='Gt(HP)'	#default grt solution model
	if [[ ! $QUIET ]]; then
		echo "Using default garnet solution model: $GRTNAME"
	fi
else
	if [[ $DEBUG ]]; then
		echo "Using garnet solution model: $GRTNAME"
	fi
fi


shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

if [[ -n "$1" ]]; then
	BUILDFILE=$1
else  
	echo "Need to specify the build file name as the last argument."
	echo '  (Leave off the .txt sufffix.)'
	exit $E_OPTERROR
fi


RUNNAME=`printf "${BUILDFILE}" | sed -E -n -e 's/([^.]*)(\.txt)/\1/p'`
if [[ $DEBUG ]]; then echo "RUNNAME: $RUNNAME"; fi
RESULTSFILE="Results.txt"
PLOTNAME="${RUNNAME}Plot"
if [[ $DEBUG ]]; then echo "PLOTNAME: $PLOTNAME"; fi
PSVDRAWINPUT=psvdraw_in.txt
PSVDRAWOUTPUT=temp_psvdraw_output.txt
VERTEXINPUT=vertex_in.txt

if [ -e $PSVDRAWINPUT ]; then rm $PSVDRAWINPUT; fi
if [ -e $PSVDRAWOUTPUT ]; then rm $PSVDRAWOUTPUT; fi
if [ -e $RESULTSFILE ]; then rm $RESULTSFILE; fi


sed -E -e '3 s/(.*)(  plot file)(.*)/'$PLOTNAME'\2\3/' $BUILDFILE > temp.txt
rm  ${RUNNAME}.txt
mv temp.txt  ${RUNNAME}.txt

echo "Results for run $RUNNAME." > $RESULTSFILE

echo "${RUNNAME}.txt" > $VERTEXINPUT

if [[ ! $NOVERTEX ]]; then
	./vertex < $VERTEXINPUT
fi

if [ ! -e "$PLOTNAME" ]
then
	echo "Plot file $PLOTNAME is missing. Quitting."
	exit
fi

if [ ! -e "./vertex" ]; then
	echo "The perplex program vertex is missing. Quitting."
	exit
fi

if [ ! -e "./psvdraw" ]
then
	echo "The perplex program psvdraw is missing. Quitting."
	exit
fi

# Otherwise, continue on:

make_one_plot () {
	TARGETBOT=$1
	TARGETTOP=$2
	PLOTTYPE=$3
	PLOTDESC=$4
	PSVDRAWPARAMS=$5

	printf $PSVDRAWPARAMS >> $PSVDRAWINPUT

	./psvdraw < $PSVDRAWINPUT >> $PSVDRAWOUTPUT
	
	PROPERTYBOTTOM=`sed -E -n -e 's/(The property ranges from:)([[:space:]]*)([[:digit:]\.]+)([[:space:]]*)->([[:space:]]*)([[:digit:]\.]+)(.*)/\3/p' ${PSVDRAWOUTPUT}`
	PROPERTYTOP=`sed -E -n -e 's/(The property ranges from:)([[:space:]]*)([[:digit:]\.]+)([[:space:]]*)->([[:space:]]*)([[:digit:]\.]+)(.*)/\6/p' ${PSVDRAWOUTPUT}`
	
	echo "$PLOTDESC ranges from $PROPERTYBOTTOM to $PROPERTYTOP." >> $RESULTSFILE
	
	if [ $TARGETBOT = "-1" ]; then
		PLOTBOTTOM=$PROPERTYBOTTOM
	else
		ANS=`echo "$TARGETBOT > $PROPERTYTOP" | bc`		# need bc for floating point comparisons
		if [ $ANS -eq 1 ]; then
			echo "Your desired bottom boundary for this plot, $TARGETBOT, is above the top value in the pseudosection, $PROPERTYTOP.  This plot cannot be made."
			return 0
		fi
	
		ANS=`echo "$PROPERTYBOTTOM > $TARGETBOT" | bc`
		if [ $ANS -eq 1 ]; then
			PLOTBOTTOM=$PROPERTYBOTTOM
		else
			PLOTBOTTOM=$TARGETBOT
		fi
	fi

	if [[ $DEBUG ]]; then echo "PLOTBOTTOM: $PLOTBOTTOM"; fi

	if [ $TARGETTOP = "-1" ]; then
		PLOTTOP=$PROPERTYTOP
	else
		ANS=`echo "$TARGETTOP < $PROPERTYBOTTOM" | bc`		# need bc for floating point comparisons
		if [ $ANS -eq 1 ]; then
			echo "Your desired top boundary for this plot, $TARGETTOP, is below the bottom value in the pseudosection, $PROPERTYBOTTOM.  This plot cannot be made."
			return 0
		fi
		
		ANS=`echo "$PROPERTYTOP < $TARGETTOP" | bc`
		if [ $ANS -eq 1 ]; then
			PLOTTOP=$PROPERTYTOP
		else
			PLOTTOP=$TARGETTOP
		fi
	fi

	if [[ $DEBUG ]]; then echo "PLOTTOP: $PLOTTOP"; fi
	
	printf "$PLOTBOTTOM \n$PLOTTOP" >> $PSVDRAWINPUT	# range of mode to plot
	
	./psvdraw < $PSVDRAWINPUT
	
	echo "Plotted $PLOTDESC ranges from $PLOTBOTTOM to $PLOTTOP." >> $RESULTSFILE
	
	BOTSTR=`printf "%1.2f\n" $PLOTBOTTOM`
	TOPSTR=`printf "%1.2f\n" $PLOTTOP`
	
	mv "$PLOTNAME.ps" "${PLOTNAME}_${PLOTTYPE}_${BOTSTR}-${TOPSTR}.ps"
}

make_one_phaseoxide_plot () {
	TARGETBOT=$1
	TARGETTOP=$2
	PHASE=$3
	PHASENAME=$4
	PARAM=$5
	PLOTTYPE="$4$5"
	PLOTDESC=$6
	PSVDRAWPARAMS="$PLOTNAME\ny\ny\nn\nn\nn\ny\nn\ny\ny\n$PHASE\n\nn\nn\nn\ny\nn\n8\n$PHASE\n1\n"
	PSVDRAWPARAMSTEMPSUFFIX="1\n1\n0\nn\n0\n0\nn\n"

	# for phase oxide plots, we assume weight percent.  Need to ensure that the perplex options are set appropriately.
	OPTIONFILENAME="perplex_option.dat"
	if [ -e $OPTIONFILENAME ]; then
		sed -E -e 's/^composition ([^ ]*)$/composition wt/' $OPTIONFILENAME > temp.txt
		rm $OPTIONFILENAME
		mv temp.txt $OPTIONFILENAME
	else
		echo 'composition wt' > perplex_option.dat
	fi


	printf $PSVDRAWPARAMS > $PSVDRAWINPUT

	printf $PSVDRAWPARAMSTEMPSUFFIX >> $PSVDRAWINPUT

	./psvdraw < $PSVDRAWINPUT > $PSVDRAWOUTPUT

	# Need to figure out the ordinal of our chosen PARAM (e.g., MNO, FEO)
	TEMPLINE=`sed -E -n -e '/(Enter component indices and weighting factors)(.*)/=' ${PSVDRAWOUTPUT}`
	PARAMNUM=`sed -E -n -e ' '${TEMPLINE}',$ s/([[:digit:]]) - '$PARAM'/\1/p' ${PSVDRAWOUTPUT}`
	
	if [[ $DEBUG ]]; then
		echo "Line number of component header: $TEMPLINE"
		echo "Ordinal of parameter $PARAM: $PARAMNUM"
	fi


	printf $PSVDRAWPARAMS > $PSVDRAWINPUT
	printf "$PARAMNUM\n1\n0\nn\n0\n0\nn\n" >> $PSVDRAWINPUT		# enter parameter number for oxide

	./psvdraw < $PSVDRAWINPUT > $PSVDRAWOUTPUT		# re-run psvdraw to get range of parameter in output
	
	PROPERTYBOTTOM=`sed -E -n -e 's/(The property ranges from:)([[:space:]]*)([[:digit:]\.E-]+)([[:space:]]*)->([[:space:]]*)([[:digit:]\.E-]+)(.*)/\3/p' ${PSVDRAWOUTPUT}`
	PROPERTYTOP=`sed -E -n -e 's/(The property ranges from:)([[:space:]]*)([[:digit:]\.E-]+)([[:space:]]*)->([[:space:]]*)([[:digit:]\.E-]+)(.*)/\6/p' ${PSVDRAWOUTPUT}`

	if [[ $DEBUG ]]; then
		echo "$PLOTDESC ranges from $PROPERTYBOTTOM to $PROPERTYTOP."
	fi
	
	echo "$PLOTDESC ranges from $PROPERTYBOTTOM to $PROPERTYTOP." >> $RESULTSFILE

	if [ $TARGETBOT = "-1" ]; then
		PLOTBOTTOM=$PROPERTYBOTTOM
	else
		ANS=`echo "$TARGETBOT > $PROPERTYTOP" | bc`		# need bc for floating point comparisons
		if [ $ANS -eq 1 ]; then
			echo "Your desired bottom boundary for this plot, $TARGETBOT, is above the top value in the pseudosection, $PROPERTYTOP.  This plot cannot be made."
			return 0
		fi
	
		ANS=`echo "$PROPERTYBOTTOM > $TARGETBOT" | bc`
		if [ $ANS -eq 1 ]; then
			PLOTBOTTOM=$PROPERTYBOTTOM
		else
			PLOTBOTTOM=$TARGETBOT
		fi
	fi

	if [[ $DEBUG ]]; then echo "PLOTBOTTOM: $PLOTBOTTOM"; fi

	if [ $TARGETTOP = "-1" ]; then
		PLOTTOP=$PROPERTYTOP
	else
		ANS=`echo "$TARGETTOP < $PROPERTYBOTTOM" | bc`		# need bc for floating point comparisons
		if [ $ANS -eq 1 ]; then
			echo "Your desired top boundary for this plot, $TARGETTOP, is below the bottom value in the pseudosection, $PROPERTYBOTTOM.  This plot cannot be made."
			return 0
		fi
		
		ANS=`echo "$PROPERTYTOP < $TARGETTOP" | bc`
		if [ $ANS -eq 1 ]; then
			PLOTTOP=$PROPERTYTOP
		else
			PLOTTOP=$TARGETTOP
		fi
	fi
	if [[ $DEBUG ]]; then echo "PLOTTOP: $PLOTTOP"; fi
	
	printf $PSVDRAWPARAMS > $PSVDRAWINPUT
	printf "$PARAMNUM\n1\n0\nn\n$PLOTBOTTOM\n$PLOTTOP\nn\n" >> $PSVDRAWINPUT		# enter range of oxide to plot
	
	./psvdraw < $PSVDRAWINPUT
	
	echo "Plotted $PLOTDESC ranges from $PLOTBOTTOM to $PLOTTOP." >> $RESULTSFILE
	
	BOTSTR=`printf "%1.2f\n" $PLOTBOTTOM`
	TOPSTR=`printf "%1.2f\n" $PLOTTOP`
	
	mv "$PLOTNAME.ps" "${PLOTNAME}_${PLOTTYPE}_${BOTSTR}-${TOPSTR}.ps"
}

make_one_phaseelement_plot () {
	# this type of plot gives [element]#, ie., Fe / (Fe + Mg + Mn + Ca)
	TARGETBOT=$1
	TARGETTOP=$2
	PHASE=$3
	PHASENAME=$4
	PARAM=$5
	OTHERELEMENTS=$6
	PLOTTYPE="$4$5"
	PLOTDESC=$7
	PSVDRAWPARAMS="$PLOTNAME\ny\ny\nn\nn\nn\ny\nn\ny\ny\n$PHASE\n\nn\nn\nn\ny\nn\n8\n$PHASE\n"
	PSVDRAWPARAMSTEMPSUFFIX="1\n1\n1\n0\nn\n0\n0\nn\n"

	# for phase element plots, we assume molar percent.  Need to ensure that the perplex options are set appropriately.
	OPTIONFILENAME="perplex_option.dat"
	if [ -e $OPTIONFILENAME ]; then
		sed -E -e 's/^composition ([^ ]*)$/composition mol/' $OPTIONFILENAME > temp.txt
		rm $OPTIONFILENAME
		mv temp.txt $OPTIONFILENAME
	else
		echo 'composition wt' > perplex_option.dat
	fi

	printf $PSVDRAWPARAMS > $PSVDRAWINPUT
	printf $PSVDRAWPARAMSTEMPSUFFIX >> $PSVDRAWINPUT

	./psvdraw < $PSVDRAWINPUT > $PSVDRAWOUTPUT

	# Need to figure out the ordinal of our chosen PARAM (e.g., MNO, FEO)
	TEMPLINE=`sed -E -n -e '/(Enter component indices and weighting factors)(.*)/=' ${PSVDRAWOUTPUT}`
	PARAMNUM=`sed -E -n -e ' '${TEMPLINE}',$ s/([[:space:]]*)([[:digit:]])([[:space:]]*)-([[:space:]]*)'$PARAM'([[:space:]]*)/\2/p' ${PSVDRAWOUTPUT}`
	
	if [[ $DEBUG ]]; then echo "Line number of component header: $TEMPLINE"; fi
	if [[ $DEBUG ]]; then echo "Ordinal of parameter $PARAM: |${PARAMNUM}|"; fi

	# figure out the ordinals of the other elements
	NUMOTHERS=`perl -e 'my $numElements=split(/_/, '$OTHERELEMENTS');print $numElements;'`
	if [[ $DEBUG ]]; then echo "NUMOTHERS: $NUMOTHERS"; fi
	PSVDRAWINPUTSTRPART2="1\n$PARAMNUM\n1\n$NUMOTHERS\n"
	for ((a=1; a <= NUMOTHERS ; a++))
	do
		THISOTHEROXIDE=`perl -e 'my @elements=split(/_/, '$OTHERELEMENTS');print $elements['$a-1'];'`
		if [[ $DEBUG ]]; then echo "THISOTHEROXIDE: $THISOTHEROXIDE"; fi
		THISOTHEROXIDENUM=`sed -E -n -e ' '${TEMPLINE}',$ s/([[:space:]]*)([[:digit:]])([[:space:]]*)-([[:space:]]*)'$THISOTHEROXIDE'([[:space:]]*)/\2/p' ${PSVDRAWOUTPUT}`
		if [[ $DEBUG ]]; then echo "THISOTHEROXIDENUM: |${THISOTHEROXIDENUM}|"; fi
		PSVDRAWINPUTSTRPART2="${PSVDRAWINPUTSTRPART2}${THISOTHEROXIDENUM}\n1\n"
		if [[ $DEBUG ]]; then echo "Current other elementsstring: $PSVDRAWINPUTSTRPART2"; fi
	done
	PSVDRAWINPUTSTRPART2="${PSVDRAWINPUTSTRPART2}n\n"
	if [[ $DEBUG ]]; then echo "Current other elementsstring: $PSVDRAWINPUTSTRPART2"; fi

	# re-run psvdraw, but now with correct property (e.g., Mn#) in order to find the max and min values of the property
	printf $PSVDRAWPARAMS > $PSVDRAWINPUT
	printf $PSVDRAWINPUTSTRPART2 >> $PSVDRAWINPUT		# enter parameter number for oxide
	printf "0\n0\nn\n" >> $PSVDRAWINPUT		# enter parameter number for oxide

	./psvdraw < $PSVDRAWINPUT > $PSVDRAWOUTPUT		# re-run psvdraw to get range of parameter in output
	
	PROPERTYBOTTOM=`sed -E -n -e 's/(The property ranges from:)([[:space:]]*)([[:digit:]\.E-]+)([[:space:]]*)->([[:space:]]*)([[:digit:]\.E-]+)(.*)/\3/p' ${PSVDRAWOUTPUT}`
	#remove any scientific notation from the number:
	PROPERTYBOTTOM=`echo "$PROPERTYBOTTOM" | awk '{ printf ( "%f\n", $0 ) }'`
	PROPERTYTOP=`sed -E -n -e 's/(The property ranges from:)([[:space:]]*)([[:digit:]\.E-]+)([[:space:]]*)->([[:space:]]*)([[:digit:]\.E-]+)(.*)/\6/p' ${PSVDRAWOUTPUT}`
	#remove any scientific notation from the number:
	PROPERTYTOP=`echo "$PROPERTYTOP" | awk '{ printf ( "%f\n", $0 ) }'`

	if [[ $DEBUG ]]; then echo "$PLOTDESC ranges from $PROPERTYBOTTOM to $PROPERTYTOP."; fi
	
	echo "$PLOTDESC ranges from $PROPERTYBOTTOM to $PROPERTYTOP." >> $RESULTSFILE

	if [[ $DEBUG ]]; then echo "PROPERTYBOTTOM: $PROPERTYBOTTOM"; fi
	if [[ $DEBUG ]]; then echo "TARGETBOT: $TARGETBOT"; fi
	if [[ $DEBUG ]]; then echo "PROPERTYTOP: $PROPERTYTOP"; fi
	if [[ $DEBUG ]]; then echo "TARGETTOP: $TARGETTOP"; fi



	if [ $TARGETBOT = "-1" ]; then
		PLOTBOTTOM=$PROPERTYBOTTOM
	else
		ANS=`echo "$TARGETBOT > $PROPERTYTOP" | bc`		# need bc for floating point comparisons
		if [ $ANS -eq 1 ]; then
			echo "Your desired bottom boundary for this plot, $TARGETBOT, is above the top value in the pseudosection, $PROPERTYTOP.  This plot cannot be made."
			return 0
		fi
	
		ANS=`echo "$PROPERTYBOTTOM > $TARGETBOT" | bc`
		if [ $ANS -eq 1 ]; then
			PLOTBOTTOM=$PROPERTYBOTTOM
		else
			PLOTBOTTOM=$TARGETBOT
		fi
	fi

	if [[ $DEBUG ]]; then echo "PLOTBOTTOM: $PLOTBOTTOM"; fi

	if [ $TARGETTOP = "-1" ]; then
		PLOTTOP=$PROPERTYTOP
	else
		ANS=`echo "$TARGETTOP < $PROPERTYBOTTOM" | bc`		# need bc for floating point comparisons
		if [ $ANS -eq 1 ]; then
			echo "Your desired top boundary for this plot, $TARGETTOP, is below the bottom value in the pseudosection, $PROPERTYBOTTOM.  This plot cannot be made."
			return 0
		fi
		
		ANS=`echo "$PROPERTYTOP < $TARGETTOP" | bc`
		if [ $ANS -eq 1 ]; then
			PLOTTOP=$PROPERTYTOP
		else
			PLOTTOP=$TARGETTOP
		fi
	fi
	if [[ $DEBUG ]]; then echo "PLOTTOP: $PLOTTOP"; fi
	
	printf $PSVDRAWPARAMS > $PSVDRAWINPUT
	printf $PSVDRAWINPUTSTRPART2 >> $PSVDRAWINPUT		# enter parameter number for oxide
	printf "${PLOTBOTTOM}\n${PLOTTOP}\nn\n" >> $PSVDRAWINPUT		# enter range of oxide to plot

	./psvdraw < $PSVDRAWINPUT
	
	echo "Plotted $PLOTDESC ranges from $PLOTBOTTOM to $PLOTTOP." >> $RESULTSFILE
	
	BOTSTR=`printf "%1.2f\n" $PLOTBOTTOM`
	TOPSTR=`printf "%1.2f\n" $PLOTTOP`
	
	mv "$PLOTNAME.ps" "${PLOTNAME}_${PLOTTYPE}#_${BOTSTR}-${TOPSTR}.ps"
}

# arguments: target bottom of range to fill, target top, plot type (for filename), plot type (for results), set of prompts for psvdraw
#make_one_plot 0.0 25.0 "GrtMode" "Garnet Mode" "$PLOTNAME\ny\nn\nn\nn\ny\nn\n7\n$GRTNAME\nn\n"
make_one_phaseoxide_plot -1 -1 $GRTNAME Grt MNO "Garnet MnO Content"
make_one_phaseelement_plot -1 -1 $GRTNAME "Grt" MNO MNO_FEO_CAO_MGO "Garnet Mn#"
make_one_phaseelement_plot -1 -1 $GRTNAME "Grt" FEO MNO_FEO_CAO_MGO "Garnet Fe#"
make_one_phaseelement_plot -1 -1 $GRTNAME "Grt" CAO MNO_FEO_CAO_MGO "Garnet Ca#"
make_one_phaseelement_plot -1 -1 $GRTNAME "Grt" MGO MNO_FEO_CAO_MGO "Garnet Mg#"
make_one_phaseelement_plot 0.47 0.48 $GRTNAME "Grt" MNO MNO_FEO_CAO_MGO "Garnet Mn#"
make_one_phaseelement_plot 0.36 0.37 $GRTNAME "Grt" FEO MNO_FEO_CAO_MGO "Garnet Fe#"
make_one_phaseelement_plot 0.09 0.10 $GRTNAME "Grt" CAO MNO_FEO_CAO_MGO "Garnet Ca#"
make_one_phaseelement_plot 0.04 0.05 $GRTNAME "Grt" MGO MNO_FEO_CAO_MGO "Garnet Mg#"

######################################
## CLEAN UP
######################################

if [ -e $PSVDRAWOUTPUT ]
then
rm $PSVDRAWOUTPUT
fi

if [ -e $PSVDRAWINPUT ]
then
rm $PSVDRAWINPUT
fi

if [ -e $VERTEXINPUT ]
then
rm $VERTEXINPUT
fi

exit 0
